# database.py

import gadfly
import os
import string
import traceback
try:
    import cPickle as pickle
except ImportError:
    import pickle
#
import date
import renderer

class Entry:
    """ Placeholder for an blogentry record. """
    def __init__(self):
        self.title = ""
        self.author = ""
        self.text = ""
        self.misc = {}

        self.anchor = ""


class Blog:
    """ Blog object to be accessed from embedded code. """
    def __init__(self, bloginfo):
        for key, value in bloginfo.get_attributes():
            setattr(self, key, value)
        self.entries = []   # must be filled by caller
        self.entry_template = ""    # ditto
        self.archives = []
    def set_entries(self, entries):
        self.entries = entries
    def messages(self): # or: entries()
        html_pieces = []
        namespace = {"blog": self}
        for entry in self.entries:
            namespace["entry"] = entry
            html = renderer.render(self.entry_template, namespace)
            html_pieces.append(html)
        return string.join(html_pieces, "")

    def archive_list(self):
        """ Generate an archive list in HTML. """
        html = []
        for pagename, ids in self.archives:
            n = string.split(pagename, ".")[0]
            parts = string.split(n, "_")
            text = string.join(parts[2:], "-")  # discard first two
            line = '<a href="%s">%s</a><br>' % (pagename, text)
            html.append(line)
        return string.join(html, "\n")

    def get_permalink(self, entry_id):
        """ Return the permalink for an entry. """
        for pagename, ids in self.archives:
            if entry_id in ids:
                return "%s#e%s" % (pagename, entry_id)
        return "permalink not found"
 

class Database:

    def __init__(self, database, new=0):
        self.name = database
        # make sure directory is available
        dirname = "db_" + database
        try:
            os.mkdir(dirname)
        except OSError, e:
            # suppress exception if directory already exists
            if not (e.args and e.args[0] == 17):
                raise

        if new:
            self.conn = gadfly.gadfly()
            self.conn.startup("database", dirname)
        else:
            self.conn = gadfly.gadfly("database", dirname)

    def create(self, database):
        cursor = self.conn.cursor()
        cursor.execute("""
         create table blogentry (
          blogentry_id integer,
          title varchar,
          text varchar,
          author varchar,
          insert_date varchar,
          modified_date varchar,
          misc varchar
         )
        """)
        self.conn.commit()

    ###
    ### entries

    def create_new_entry_id(self):
        """ Create a new entry id. """
        cursor = self.conn.cursor()
        try:
            cursor.execute("""
             select max(blogentry_id) as maxid
             from blogentry
            """)
            rows = cursor.fetchall()
        except ValueError:
            # XXX unsafe; what if we get a ValueError for other reasons than an
            # empty table?
            return 1
        return rows[0][0] + 1

    def insert_entry(self, entry):
        now = date.Date().isodate()
        newid = self.create_new_entry_id()
        cursor = self.conn.cursor()
        cursor.execute("""
         insert into blogentry (blogentry_id, title, text, author, insert_date,
          modified_date, misc)
         values (?, ?, ?, ?, ?, ?, ?)
        """, (newid, entry.title, entry.text, entry.author, now, now, 
              repr(entry.misc)))
        self.conn.commit()
        return newid

    def delete_entry(self, id):
        cursor = self.conn.cursor()
        cursor.execute("""
         delete from blogentry
         where blogentry_id = ?
        """, (id,))
        self.conn.commit()

    def update_entry(self, entry):
        """ Update an existing entry. This entry is assumed to have an entry_id
            field. """
        now = date.Date().isodate()
        cursor = self.conn.cursor()
        cursor.execute("""
         update blogentry
         set title = ?,
          text = ?,
          author = ?,
          misc = ?,
          modified_date = ?
         where blogentry_id = ?
        """, (entry.title, entry.text, entry.author, repr(entry.misc), 
              now, entry.blogentry_id))
        self.conn.commit()

    def get_entry(self, id):
        """ Get an entry record and return it as a dictionary. """
        cursor = self.conn.cursor()
        cursor.execute("""
         select * from blogentry
         where blogentry_id = ?
        """, (id,))

        return self._make_entry_objects(cursor)[0]

    def get_entries_by_id(self, start_id, end_id, visible_only=0):
        cursor = self.conn.cursor()
        cursor.execute("""
         select * from blogentry
         where blogentry_id >= ?
         and blogentry_id <= ?
         order by blogentry_id desc
        """, (start_id, end_id))
        
        entries = self._make_entry_objects(cursor)
        if visible_only:
            entries = [e for e in entries if e.misc.get("visible", 1)]
        return entries

    def get_entries_by_idlist(self, ids, visible_only=0):
        if not ids:
            return []
        if len(ids) == 1:
            ids.append(ids[0])  # avoid SQL error
        cursor = self.conn.cursor()
        idlist = repr(tuple(ids))[1:-1]
        cursor.execute("""
         select * from blogentry
         where blogentry_id in (%s)
         order by blogentry_id desc
        """ % (idlist,))

        entries = self._make_entry_objects(cursor)
        if visible_only:
            entries = [e for e in entries if e.misc.get("visible", 1)]
        return entries

    def get_entry_dates(self, visible_only=0):
        """ Get all entries, but only return the id and the date. """
        cursor = self.conn.cursor()
        cursor.execute("""
         select blogentry_id, insert_date, misc
         from blogentry
         order by blogentry_id desc
        """)
        entries = self._make_entry_objects(cursor)
        if visible_only:
            entries = [e for e in entries if e.misc.get("visible", 1)]
        return entries

    def get_entries_by_date(self, start_date, end_date, visible_only=0):
        cursor = self.conn.cursor()
        cursor.execute("""
         select * from blogentry
         where insert_date >= ?
         and insert_date <= ?
         order by blogentry_id desc
        """, (start_date, end_date))

        entries = self._make_entry_objects(cursor)
        if visible_only:
            entries = [e for e in entries if e.misc.get("visible", 1)]
        return entries

    def get_most_recent_entries(self, number, visible_only=0):
        # for now, we simply do this by getting all entries and getting the
        # topmost <number> of them. later we might opt for something more
        # efficient.
        entries = self.get_entries_by_date("2000-01-01", "2099-31-12")
        if visible_only:
            entries = [e for e in entries if e.misc.get("visible", 1)]
        return entries[:number]

    def get_entries_by_category(self, number, category, visible_only=0):
        # get blogentry_id and misc fields so query won't be slower than
        # necessary
        cursor = self.conn.cursor()
        cursor.execute("""
         select blogentry_id, misc
         from blogentry
         where blogentry.misc <> ?
        """, ("{}",))
        entries = self._make_entry_objects(cursor)

        # select all entries that are in the given category
        ids = []
        for e in entries:
            if e.misc:
                if category in e.misc.get("categories", []):
                    ids.append(e.blogentry_id)
            
        ids = ids[-number:]
        print "ids found for", category, ":", ids
        return self.get_entries_by_idlist(ids, visible_only)

    ###
    ### helper methods

    def _make_entry_objects(self, cursor):
        """ Take a cursor with a recordset, and convert the records to Entry
            instances. """
        fieldnames = [r[0].lower() for r in cursor.description]
        rows = []
        for row in cursor.fetchall():
            entry = Entry()
            for i in range(len(row)):
                setattr(entry, fieldnames[i], row[i])
            if entry.misc:
                entry.misc = eval(entry.misc)
            else:
                entry.misc = {}
            rows.append(entry)
        return rows

    ###
    ### statistics

    def get_num_entries(self):
        """ Return the number of entries in the blog. """
        cursor = self.conn.cursor()
        cursor.execute("""
         select count(blogentry_id) as c from blogentry
        """)
        for row in cursor.fetchall():
            count = row[0]
            return count

    ###
    ### dumping database contents

    def dump_pickled(self, filename):
        f = open(filename, "wb")
        entries = self.get_entries_by_date("2000-01-01", "2099-31-12")
        for entry in entries:
            pickle.dump(entry, f)
        f.close()

    def dump_xml(self, filename):
        pass
        # XXX yet to be written!
