# kaa.py
#
#----------------------------------------------------------------------------
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#----------------------------------------------------------------------------
# File creation date: 28 Jul 2002
# (C) 2002 Hans Nowak

import sys
import os
import getopt
import string
import types
import webbrowser
from Tkinter import *
import tkSimpleDialog
#
import archiver
import bloginfo_input
import category_entry_input
import category_input
import categorymanager
import cm
import configuration
import database
import date
import entrypicker
import errors
import ftpinfo_input
import htmlmaker
import htmltools
import htmlwriter
import publisher
import resource
import rsswriter
import statusbar
import tools

class Kaa:

    def __init__(self, master, blogname=None):
        frame = self.frame = Frame(master)
        
        # set Tkinter's exception hook
        Tk.report_callback_exception = self.excepthook

        self.menu = self.create_menu(frame)
        master.config(menu=self.menu)

        self.toolbar1 = self.create_toolbar_1(frame)
        self.toolbar1.pack(side=TOP, fill=X)

        self.toolbar2 = self.create_toolbar_2(frame)
        self.toolbar2.pack(side=TOP, fill=X)

        self.titlebar = Text(frame, height=1, width=70,
         font=resource.FIXED_FONT)
        self.titlebar.pack(side=TOP, padx=1, pady=1, fill=X)

        self.editorframe = self.create_scrolledtext(frame, width=70, height=20)
        self.editor = self.editorframe.text
        self.editorframe.pack(side=TOP, padx=1, pady=1, fill=BOTH, expand=YES)

        self.statusbar = self.create_statusbar(frame)
        self.statusbar.pack(side=TOP, fill=X)
        self.statusbar.set_text(1, "No blog loaded.")

        frame.pack(side=TOP, fill=BOTH, expand=YES)

        self.titlebar.bind("<Tab>", self.set_editor_focus)  
        self.editor.bind("<Tab>", self.set_titlebar_focus)
        self.load_editor_bindings()

        self.db = None  # for now; must be set by open_blog or create_blog
        self.current_entry = None   # current Entry instance
        #self.categories = []
        self.categorymanager = None
        self.pagegens = None    # custom page generators
        self.globalmod = None   # global variables for embedded code

        if blogname:
            print "Opening blog:", blogname
            self._open_blog(blogname)

        self.titlebar.focus_set()

    def create_toolbar_1(self, master):
        frame = Frame(master, relief=GROOVE, borderwidth=1)

        new_button = Button(frame, text="New entry", command=self.new_entry,
         font=resource.SMALL_FONT)
        new_button.pack(side=LEFT)
        
        save_button = Button(frame, text="Save", command=self.save_entry, 
         font=resource.SMALL_FONT)
        save_button.pack(side=LEFT)

        post_button = Button(frame, text="Post", command=self.post, 
         font=resource.SMALL_FONT)
        post_button.pack(side=LEFT)

        publish_button = Button(frame, text="Publish", command=self.publish_all)
        publish_button.configure(font=resource.SMALL_FONT)
        publish_button.pack(side=LEFT)

        separator = Frame(frame, width=3)
        separator.pack(side=LEFT)

        b = Button(frame, text="One-click publishing",
         command=self.one_click_publish, font=resource.SMALL_FONT)
        b.pack(side=LEFT)
        
        separator = Frame(frame, width=3)
        separator.pack(side=LEFT)

        edit_categories_entry_button = Button(frame, text="Categories",
         command=self.edit_categories_entry, font=resource.SMALL_FONT)
        edit_categories_entry_button.pack(side=LEFT)

        self.quit_button = Button(frame, text="quit", fg="red", 
         command=master.quit, font=resource.SMALL_FONT)
        self.quit_button.pack(side=RIGHT)

        return frame

    def create_toolbar_2(self, master):
        """ Create a HTML toolbar. """
        frame = Frame(master, relief=GROOVE, borderwidth=1)

        def methodmaker(self, f):
            return lambda: f(self)

        # load buttons from file
        import config_actions
        
        for text, action in config_actions.buttons:
            if isinstance(action, str):
                action = eval(action)
            else:
                action = methodmaker(self, action)
            button = Button(frame, text=text, command=action,
             font=resource.SMALL_FONT)
            button.pack(side=LEFT)

        return frame

    def create_statusbar(self, master):
        # create a statusbar with two panels of width 50 and 30
        sb = statusbar.StatusBar(master, [50, 30])
        return sb

    def create_scrolledtext(self, master, width, height):
        f = Frame(master)
        f.pack(side=TOP, expand=YES, fill=BOTH)

        scrollbar = Scrollbar(f, orient="vertical")
        scrollbar.pack(side=RIGHT, fill=Y)
        
        text = Text(f, width=width, height=height)
        text.configure(font=resource.FIXED_FONT, wrap=WORD)
        text.pack(side=LEFT, fill=BOTH, expand=YES)
        text["yscrollcommand"] = scrollbar.set
        scrollbar["command"] = text.yview

        f.text = text
        f.scrollbar = scrollbar

        return f

    def create_menu(self, master):
        menubar = Menu(master)
        
        blogmenu = Menu(menubar, tearoff=0)
        blogmenu.add_command(label="New...", command=self.create_blog)
        blogmenu.add_command(label="Open...", command=self.open_blog)
        blogmenu.add_separator()
        blogmenu.add_command(label="Edit blog info",
         command=self.edit_blog_info)
        blogmenu.add_command(label="Edit FTP info", command=self.edit_ftp_info)
        blogmenu.add_command(label="Edit categories",
         command=self.edit_categories)
        blogmenu.add_separator()
        blogmenu.add_command(label="Pickle database",
         command=self.dump_pickled_blog)
        blogmenu.add_separator()
        blogmenu.add_command(label="Exit", command=master.quit)
        blogmenu.add_command(label="(Test exception)", 
         command=self.test_exception)
        menubar.add_cascade(label="Blog", menu=blogmenu)

        entrymenu = Menu(menubar, tearoff=0)
        entrymenu.add_command(label="Create new entry", command=self.new_entry)
        entrymenu.add_command(label="Edit most recent entry",
         command=self.edit_last_entry)
        entrymenu.add_command(label="Save", command=self.save_entry)
        entrymenu.add_command(label="Edit existing entry",
         command=self.select_old_entry)
        entrymenu.add_command(label="Delete existing entry", 
         command=self.delete_entry)  # for now...
        entrymenu.add_separator()
        entrymenu.add_command(label="Show attributes",
         command=self.show_attributes)
        entrymenu.add_command(label="Edit categories",
         command=self.edit_categories_entry)
        entrymenu.add_command(label="Make visible", command=self.make_visible)
        entrymenu.add_command(label="Make invisible",
         command=self.make_invisible)
        menubar.add_cascade(label="Entry", menu=entrymenu)

        publishmenu = Menu(menubar, tearoff=0)
        publishmenu.add_command(label="Post", command=self.post)
        publishmenu.add_command(label="Preview", command=self.preview)
        publishmenu.add_command(label="Publish all", command=self.publish_all)
        # to be added: other publish methods, etc
        menubar.add_cascade(label="Publishing", menu=publishmenu)

        editmenu = Menu(menubar, tearoff=0)
        editmenu.add_command(label="Bold", command=self.wrap_bold)
        editmenu.add_command(label="Italic", command=self.wrap_italic)
        editmenu.add_command(label="Underline", command=self.wrap_underline)
        editmenu.add_separator()
        editmenu.add_cascade(label="Insert URL", command=self.wrap_url)
        editmenu.add_cascade(label="Quote HTML", command=self.quote_text)
        editmenu.add_cascade(label="Unquote HTML", command=self.unquote_text)
        menubar.add_cascade(label="Edit", menu=editmenu)
        
        return menubar

    ###
    ### blogs

    def open_blog(self):
        name = tkSimpleDialog.askstring("Enter blog name", "Name of blog")
        if name:
            self._open_blog(name)

    def create_blog(self):
        name = tkSimpleDialog.askstring("Enter blog name", "Name of blog")
        if name:
            print "Creating blog '%s'..." % (name),
            self._create_blog(name)
            print "OK"
            self._open_blog(name)
            print "New blog opened."
        
    def _open_blog(self, blogname):
        """ Open a blog. """
        self.db = database.Database(blogname)
        print self.db.get_num_entries(), "entries in this blog."
        self.load_configuration_files(blogname)
        self.macros = self.load_html_macros(blogname)
        #self.categories = self.load_categories(blogname)
        self.categorymanager = self.load_categories(blogname)
        self.statusbar.set_text(1, "Active blog: %s" % (blogname,))
        
        try:
            self.pagegens = tools.import_path(os.path.join("db_" + blogname,
             "pagegen.py"))
        except ImportError:
            pass    # let this pass silently, no need for error message

        try:
            self.globalmod = tools.import_path(os.path.join("db_" + blogname,
             "globals.py"))
        except ImportError:
            print "File globals.txt not found, continue..."

    def load_configuration_files(self, blogname):
        # make sure configuration files are there
        dirname = "db_" + blogname
        
        config_blog_filename = os.path.join(dirname, "config_blog.py")
        self.bloginfo = configuration.BlogInfo()
        if not os.path.exists(config_blog_filename):
            self.bloginfo.save(config_blog_filename)
        else:
            self.bloginfo.load(config_blog_filename)
            
        config_ftp_filename = os.path.join(dirname, "config_ftp.py")
        self.ftpinfo = configuration.FTPInfo()
        if not os.path.exists(config_ftp_filename):
            self.ftpinfo.save(config_ftp_filename)
        else:
            self.ftpinfo.load(config_ftp_filename)

        stats_filename = os.path.join(dirname, "stats.py")
        self.stats = configuration.Stats()
        if not os.path.exists(stats_filename):
            self.stats.save(stats_filename)
        else:
            self.stats.load(stats_filename)
        
    def _create_blog(self, blogname):
        self.db = database.Database(blogname, new=1)
        self.db.create(database)
        # TODO: check that database doesn't already exist, otherwise it may
        # overwrite it

        self.load_configuration_files(blogname) # create or load
        self.statusbar.set_text(1, "Active blog: %s" % (blogname,))

    def edit_blog_info(self):
        """ Edit the information of the current blog. """
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
        bi = bloginfo_input.BloginfoInput(self.frame, self.bloginfo,
         "Enter blog information")
        if bi.result == "ok":
            print "Saving blog info...",
            bi.save()
            print "OK"

    def edit_ftp_info(self):
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
        fi = ftpinfo_input.FTPInfoInput(self.frame, self.ftpinfo,
         "Enter FTP server information")
        if fi.result == "ok":
            print "Saving FTP info...",
            fi.save()
            print "OK"

    def edit_categories(self):
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
        categories = self.categorymanager.categories[:]
        ci = category_input.CategoryInput(self.frame, self.categorymanager)
        if ci.result == "ok":
            if ci.categorymanager.categories != categories:
                #self.categorymanager.categories = ci.categories
                self.write_categories(self.db.name)
                print "New categories were saved."
                # reload current entry so changes are displayed
                self.reload()

    def dump_pickled_blog(self):
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
        print "Pickling entries...",
        filename = "pickled_" + self.db.name + ".txt"
        self.db.dump_pickled(filename)
        print "OK"

    ###
    ### entries

    def save_entry(self):
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
        
        print "Saving entry...",
        if self.current_entry is None:
            # make a new Entry and store it
            entry = Entry()
            entry.title = self.get_title()
            entry.text = self.get_text()
            entry.author = self.get_author()
            entry.misc = {}
            id = self.db.insert_entry(entry)
            entry.blogentry_id = id
            self.current_entry = entry
        else:
            # update the existing entry
            self.current_entry.title = self.get_title()
            self.current_entry.text = self.get_text()
            self.current_entry.author = self.get_author()   # ...?
            self.db.update_entry(self.current_entry)
        print "OK"
        self.statusbar.set_text(0, "Entry #%s was saved on %s." % (
         self.current_entry.blogentry_id, 
         date.Date().isodate()))

    def new_entry(self):
        """ Create a new entry. """
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
            
        # clear editor and title bar
        self.show_entry(-1)

    def edit_last_entry(self):
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return

        id = self.db.create_new_entry_id() - 1
        if id:
            self.show_entry(id)
            #entry = self.db.get_entry(id)
            #self.titlebar.delete(1.0, END)
            #self.titlebar.insert(END, entry.title)
            #self.editor.delete(1.0, END)
            #self.editor.insert(END, entry.text)
            #self.current_entry = entry
        else:
            # there is no last entry
            # XXX show a message stating this
            self.new_entry()

    def select_old_entry(self):
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
    
        ep = entrypicker.EntryPicker(self.frame, self.db)
        id = ep.selected_id
        if id > 0:
            self.show_entry(id)
            #entry = self.db.get_entry(id)
            #self.titlebar.delete(1.0, END)
            #self.titlebar.insert(END, entry.title)
            #self.editor.delete(1.0, END)
            #self.editor.insert(END, entry.text)
            #self.current_entry = entry

    def delete_entry(self):
        """ Let user select an entry and delete it. """
        # XXX maybe an "are you sure" dialog would be useful
        # also, multiple select is useful
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
        
        ep = entrypicker.EntryPicker(self.frame, self.db)
        id = ep.selected_id
        if id >= 0:
            print "Deleting entry...",
            self.db.delete_entry(id)
            print "OK"
            
    def reload(self):
        if self.current_entry:
            id = self.current_entry.blogentry_id
            self.show_entry(id)
            
    def show_entry(self, id=-1):
        """ Load the entry with the given id, show it, and set 
            self.current_entry. """
        if id == -1:
            self.titlebar.delete(1.0, END)
            self.editor.delete(1.0, END)
            self.current_entry = None
            self.titlebar.focus_set()
        else:
            entry = self.db.get_entry(id)
            self.titlebar.delete(1.0, END)
            self.titlebar.insert(END, entry.title)
            self.editor.delete(1.0, END)
            self.editor.insert(END, entry.text)
            self.current_entry = entry

    ###
    ### getting entry info

    def get_title(self):
        return self.titlebar.get(1.0, END).strip()

    def get_text(self):
        return self.editor.get(1.0, END)

    def get_author(self):
        return "anonymous"  # XXX for now

    ###
    ### posting

    def post(self):
        """ Post archives *and* front page. """
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return

        blog = database.Blog(self.bloginfo)
        arch = archiver.Archiver(self.db)
        self.prepare_blog(blog)
        maker = htmlmaker.HTMLMaker(blog, blog.page_template_name,
         blog.entry_template_name, self.macros, self.globalmod)
        writer = htmlwriter.HTMLWriter()
        # XXX y'know, we COULD just pass the blog instance... the maker object
        # can figure out the template names from there. And it would not need
        # quite a few other function params either...

        pages = []

        # generating archives
        if self.bloginfo.archive_method.strip():
            # XXX is the strip() still necessary?
            spam = maker.make_archives(self.db, visible_only=1)
            pages.extend(spam)
            
        try:
            posts_per_page = int(self.bloginfo.num_entries_fp)
        except:
            posts_per_page = 20 # default value

        # generating category archives
        if tools.strbool(self.bloginfo.category_archives):
            spam = maker.make_category_pages(self.categorymanager.categories, 
                   self.db, posts_per_page)
            pages.extend(spam)

        # generating front page
        entries = self.db.get_most_recent_entries(posts_per_page, 
         visible_only=1)
        blog.current_page = self.bloginfo.filename
        # set anchor attribute for first page; anchors refer to archives
        #self._set_anchors(entries, blog, arch)
            
        html = maker.make_page(entries)
        pages.append((self.bloginfo.filename, html))

        # custom pages
        if self.pagegens:
            spam = self.make_custom_pages(blog)
            pages.extend(spam)

        for pagename, html in pages:
            print "Writing:", pagename, "...",
            writer.write_page(pagename, html, self.db,
             update_only=tools.strbool(self.bloginfo.only_post_updated))
            print "OK"
            
        # RSS
        
        if tools.strbool(self.bloginfo.generate_rss):
            print "Writing RSS..."
            rw = rsswriter.RSSWriter(blog, entries)
            # generate "terse" RSS
            rssdata = rw.generate_rss()
            rssname = "%s.rss" % (self.db.name,)
            writer.write_page(rssname, rssdata, self.db, update_only=
             tools.strbool(self.bloginfo.only_post_updated))
            # generate "full" RSS
            rssdata = rw.generate_rss(verbose=1)
            rssname = "%s_full.rss" % (self.db.name,)
            writer.write_page(rssname, rssdata, self.db, update_only=
             tools.strbool(self.bloginfo.only_post_updated))

        print "OK"

    def OBSOLETE_set_anchors(self, entries, blog, archiver):
        # set anchor attribute for first page; anchors refer to archives
        for entry in entries:
            if self.bloginfo.archive_method:
                for arch_id, ids in blog.entry_blocks:
                    if entry.blogentry_id in ids:
                        pagename = archiver.get_page_name(
                         self.bloginfo.archive_method, arch_id)
                        entry.anchor = "%s#e%s" % (pagename, entry.blogentry_id)
                        break   # out of inner loop
            else:
                entry.anchor = "#e%s" % (entry.blogentry_id)

    def make_custom_pages(self, blog):
        maker = htmlmaker.HTMLMaker(blog, globalmod=self.globalmod)
        pages = []
        for pgclass in self.pagegens.__dict__.values():
            if type(pgclass) == types.ClassType \
            and issubclass(pgclass, cm.CM):
                gen = pgclass(self.globalmod)
                html = gen.gen_html(self.db, blog)
                #maker.write_page(gen.target, html, self.db,
                # update_only=int(self.bloginfo.only_post_updated))
                pages.append((gen.target, html))
        return pages

    def preview(self):
        """ Preview the first page in the default browser. """
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
            
        dirname = "html_" + self.db.name
        fullname = os.path.join(dirname, self.bloginfo.filename)
        browser = webbrowser.WindowsDefault()
        browser.open(fullname, new=1)

    ###
    ### publishing

    def publish_all(self):
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
            
        dir = "html_" + self.db.name
        if not os.path.exists(dir):
            os.mkdir(dir)
        pub = publisher.Publisher(self.db, self.ftpinfo, self.stats)
        new_only = tools.strbool(self.bloginfo.only_publish_updated)
        pub.publish_dir(dir, new_only=new_only)

    def one_click_publish(self):
        """ Save, post and publish. """
        if self.db == None:
            errors.ErrorDialog(self.frame, message="No blog loaded.")
            return
            
        try:
            self.save_entry()
            self.post()
            self.publish_all()
        except:
            raise

    ###
    ### text manipulation

    def replace_selected(self, f, *args, **kwargs):
        """ Get selected text, apply function f(text) to it, and place it back.
            This is a helper function that is called by e.g. wrap_bold, or
            anything that needs to do some text manipulation.
        """
        selection = self.editor.tag_ranges(SEL)
        if len(selection) == 0:
            errors.ErrorDialog(self.frame, message="No text selected.")
            return
        text = self.editor.get(*selection)
        text = f(text, *args, **kwargs)
        self.editor.delete(*selection)
        self.editor.insert(selection[0], text)
        
    def wrap_bold(self):
        self.replace_selected(htmltools.wrap, "<b>", "</b>")

    def wrap_italic(self):
        self.replace_selected(htmltools.wrap, "<i>", "</i>")

    def wrap_underline(self):
        self.replace_selected(htmltools.wrap, "<u>", "</u>")

    def wrap_something(self, begin, end):
        """ Helper function for customizable buttons and keys. """
        self.replace_selected(htmltools.wrap, begin, end)

    def wrap_url(self):
        url = tkSimpleDialog.askstring("Enter URL", "URL ::")
        if url:
            ahref = '<a href="%s">' % (url)
            # XXX this can be done better later, e.g. enter a possible ALT tag
            # as well... better style ^_^
            self.replace_selected(htmltools.wrap, ahref, "</a>")

    def quote_text(self):
        self.replace_selected(htmltools.quote)

    def unquote_text(self):
        self.replace_selected(htmltools.unquote)

    ###
    ### entry attributes

    def show_attributes(self):
        if not self.current_entry:
            errors.ErrorDialog(self.frame, message="There is no current entry.")
            return
    
        print "Attributes:", self.current_entry.misc

    def _set_visible(self, visible=1):
        if not self.current_entry:
            errors.ErrorDialog(self.frame, message="There is no current entry.")
            return
        self.current_entry.misc["visible"] = visible
        # don't forget to save...

    def make_visible(self):
        self._set_visible(1)

    def make_invisible(self):
        self._set_visible(0)

    def edit_categories_entry(self, event=None):
        if not self.current_entry:
            errors.ErrorDialog(self.frame, message="There is no current entry.")
            return
        assoc = self.current_entry.misc.get("categories", [])
        categories = self.categorymanager.categories[:]
        for cat in assoc:
            categories.remove(cat)  # synchronize with assoc
        ecei = category_entry_input.CategoryEntryInput(self.frame,
         categories, assoc)
        if ecei.result == "ok":
            # print categories, assoc
            if self.current_entry.misc.get("categories", []) != assoc:
                self.current_entry.misc["categories"] = assoc

    ###
    ### simple HTML macros

    def load_html_macros(self, blogname):
        dirname = "db_" + blogname
        fullname = os.path.join(dirname, "htmlmacros.txt")
        macros = []
        try:
            f = open(fullname, "r")
        except IOError:
            print "File htmlmacros.txt could not be opened."
            # This file is optional, so we don't display a message
            return []
        for line in f.readlines():
            if line.strip():
                t4 = eval(line)
                macros.append(t4)
        f.close()
        return macros

    ###
    ### error handling

    def error(self, message):
        print "***", message, "***" # FIXME

    ###
    ### helper methods

    def load_editor_bindings(self):
        """ Load keybindings for the editor from the config_action.py file. """
        import config_actions
        for text, action in config_actions.keys:
            if isinstance(action, str):
                command = eval(action)
                action = lambda s=self, e=None, cmd=command: cmd()
            else:
                action = lambda s=self, e=None, cmd=action: cmd(s)
            self.editor.bind(text, action)

    def load_categories(self, blogname):
        dirname = "db_" + blogname
        filename = os.path.join(dirname, "categories.txt")
        cm = categorymanager.CategoryManager(self.db)
        try:
            cm.load(filename)
        except IOError:
            errors.ErrorDialog(self.frame, 
             message="File categories.txt could not be opened.")
        return cm

    def write_categories(self, blogname):
        dirname = "db_" + blogname
        filename = os.path.join(dirname, "categories.txt")
        self.categorymanager.save(filename)

    def set_editor_focus(self, event=None):
        self.editor.focus_set()
        return "break"  # prevent Tab character from being inserted

    def set_titlebar_focus(self, event=None):
        self.titlebar.focus_set()
        return "break"

    def set_statustext(self, text):
        self.statuslabel.configure(text=text)

    def prepare_blog(self, blog):
        """ Prepare a Blog instance for posting and publishing. """
        blog.categories = self.categorymanager.categories
        blog.info = self.bloginfo

        arch = archiver.Archiver(self.db)
        blog.entry_blocks = arch.make_archive(blog.info.archive_method, 
         visible_only=1)
        # what we need now is the filenames...

        blog.archives = [
         (arch.get_page_name(blog.info.archive_method, adate), ids)
         for (adate, ids) in blog.entry_blocks
        ]
 
        # XXX current_page & friends cannot be set here, so it must be set when
        # actually generating a page, in a controlled manner... basically, only
        # the archive maker needs previous and next pages. the "last N entries"
        # pages do not.
        
    def excepthook(self, exctype, value, traceback):
        print 'excepthook called'
        errors.ExceptionDialog(self.frame)
        
    def test_exception(self, event=None):
        # !!! this is the way to do it!!!
        #Tk.report_callback_exception = self.excepthook
        print 1/0
        
        

if __name__ == "__main__":

    blogname = ""

    opts, args = getopt.getopt(sys.argv[1:], "")

    if args:
        blogname = args[0]

    root = Tk()

    kaa = Kaa(root, blogname=blogname)
    root.title(":: Kaa %s ::" % (resource.__version__))

    root.mainloop()

