# editor.py

from wxPython.wx import *
from wxPython.stc import *
import os
#
import lexers

class Editor(wxStyledTextCtrl):

    def __init__(self, parent, options):
        self.id = wxNewId()
        wxStyledTextCtrl.__init__(self, parent, self.id, size=(480,400))
        self.filename = None    # file we're currently editing
        self.options = options

        self.set_options()
        self.set_default_style()

        # set line number stuff
        #self.StyleSetSpec(wxSTC_STYLE_LINENUMBER,
        # "fore:#000000,back:#C0C0C0,face:Tahoma,size:8")
        #self.SetMarginType(0, wxSTC_MARGIN_NUMBER)
        #if self.options.show_line_numbers:
        #    self.SetMarginWidth(0, 22)
        #    self.SetMarginType(2, wxSTC_MARGIN_SYMBOL)
        #self.SetCaretForeground(wxColour(0xE0, 0xE0, 0xE0))
        # XXX caret color should be an option later
        #self.SetCaretWidth(2)
        self.set_style("")
        

        EVT_KEY_UP(self, self.EvtKeyUp)
        EVT_KEY_DOWN(self, self.EvtKeyDown)
        #EVT_STC_UPDATEUI(self, self.id, self.EvtPosChanged)
        EVT_STC_PAINTED(self, self.id, self.EvtPosChanged)
        
        self.find_results = ("", 0) # string, position

    ###
    ### file routines

    def open(self, filename, readonly=0):
        f = open(filename, "rb")
        data = f.read()
        f.close()
        self.ClearAll()
        self.AddText(data)
        self.filename = filename
        self.SetReadOnly(readonly)
        self.SendMsg(wxSTC_CMD_DOCUMENTSTART)

        self.set_style(filename)

    def save(self, filename):
        has_filename = not not self.filename
        self.save_as(filename, take_new_filename=has_filename)

    def save_as(self, filename, take_new_filename=1):
        # XXX maybe we need to convert line endings here...
        data = self.GetText()
        f = open(filename, "wb")
        f.write(data)
        f.close()
        if take_new_filename:
            self.filename = filename
            self.set_style(filename)

    ###
    ### styles

    def set_default_style(self):
        self.SetLexer(wxSTC_LEX_CONTAINER)
        self.StyleSetSpec(wxSTC_STYLE_DEFAULT,
         "size:%d,face:%s,fore:#B0B0C0,back:#000000" % (10, "FixedSys"))

    def set_style(self, filename):
        name, ext = os.path.splitext(filename.lower())
        print "Your extension is", ext
        if self.options.file_assoc.has_key(ext):
            stylename = self.options.file_assoc[ext]
            print "Your style is", stylename
            self.StyleClearAll()
            self.SetLexer(lexers.lexers[stylename])
            self.SetKeyWords(0, string.join(lexers.lex_keywords[stylename]))

            for k, v in self.options.syntax_highlighting[stylename].items():
                c = lexers.lex_elements[stylename][k]
                self.StyleSetSpec(c, v)
        else:
            self.StyleClearAll()
            self.SetLexer(wxSTC_LEX_CONTAINER)
            self.set_default_style()

        # will this help?
        self.StyleSetSpec(wxSTC_STYLE_LINENUMBER,
         "fore:#000000,back:#C0C0C0,face:Tahoma,size:8")
        if self.options.show_line_numbers:
            self.SetMarginWidth(0, 27)
            self.SetMarginType(0, wxSTC_MARGIN_NUMBER)
        self.set_folding(self.options.folding)

        self.SetCaretForeground(wxColour(0xE0, 0xE0, 0xE0))
        # XXX caret color should be an option later
        self.SetCaretWidth(2)
                

    ###
    ### indentation

    def _get_indent(self, line):
        """ Get the indentation, as a string, from the given line. """
        # XXX slow algorithm, fix later (regex?)
        indent = ""
        for c in line:
            if c in " \t":
                indent = indent + c
            else:
                break
        return indent

    def handle_autoindent(self):
        prevlineno = self.GetCurrentLineNo() - 1
        prevline = self.GetLine(prevlineno)
        indent = self._get_indent(prevline)
        currpos = self.GetCurrentPos()
        self.InsertText(currpos, indent)
        #self.cursor_end()
        self.GotoPos(currpos + len(indent))
        
    def EvtKeyUp(self, event):
        keycode = event.GetKeyCode()
        if keycode == 13:
            self.handle_autoindent()
        # override Ctrl-PgUp and Ctrl-PgDn so they do what I want
        elif keycode == WXK_PRIOR and event.ControlDown():
            self.SendMsg(wxSTC_CMD_DOCUMENTSTART)
        elif keycode == WXK_NEXT and event.ControlDown():
            self.SendMsg(wxSTC_CMD_DOCUMENTEND)
        else:
            event.Skip()
            
    def EvtKeyDown(self, event):
        if event.GetKeyCode() == 27:
            mainframe = self.GetGrandParent()
            mainframe.set_focus_command_line()
        else:
            event.Skip()

    def GetCurrentLineNo(self):
        return self.LineFromPosition(self.GetCurrentPos())

    ###
    ### cursor routines

    def cursor_end(self):
        self.SendMsg(wxSTC_CMD_LINEEND)

    def cursor_home(self):
        """ Move the cursor to the beginning of the current line. """
        self.SendMsg(wxSTC_CMD_HOME)

    def cursor_right(self):
        """ Move the cursor to the right. """
        self.SendMsg(wxSTC_CMD_CHARRIGHT)

    def cursor_left(self):
        """ Move the cursor to the left. """
        self.SendMsg(wxSTC_CMD_CHARLEFT)

    def EvtPosChanged(self, event):
        pos = self.GetCurrentPos()
        x = self.GetColumn(pos) + 1
        y = self.LineFromPosition(pos) + 1
        # use the callback function to display coords in statusbar
        callback = self.GetParent().coords_callback
        if callback:
            callback(x, y)

    ###
    ### misc

    def set_options(self):
        self.SetUseTabs(self.options.use_tabs)
        self.SetTabWidth(self.options.tabsize)
        self.SetIndent(self.options.tabsize)
        self.SetViewWhiteSpace(self.options.show_whitespace)
        self.SetIndentationGuides(self.options.show_indentguides)
        self.set_folding(self.options.folding)
        if self.options.edge:
            self.SetEdgeMode(wxSTC_EDGE_LINE)
            self.SetEdgeColumn(self.options.edge)
        else:
            self.SetEdgeMode(wxSTC_EDGE_NONE)
        
    def set_folding(self, active=1):
        print 'setting folding...', active
        if active:
            self.SetProperty("fold", "1")
            self.SetMarginType(2, wxSTC_MARGIN_SYMBOL)
            self.SetMarginMask(2, wxSTC_MASK_FOLDERS)
            self.SetMarginSensitive(2, true)
            self.SetMarginWidth(2, 12)
            
            self.MarkerDefine(wxSTC_MARKNUM_FOLDEREND,     wxSTC_MARK_BOXPLUSCONNECTED,  "white", "black")
            self.MarkerDefine(wxSTC_MARKNUM_FOLDEROPENMID, wxSTC_MARK_BOXMINUSCONNECTED, "white", "black")
            self.MarkerDefine(wxSTC_MARKNUM_FOLDERMIDTAIL, wxSTC_MARK_TCORNER,  "white", "black")
            self.MarkerDefine(wxSTC_MARKNUM_FOLDERTAIL,    wxSTC_MARK_LCORNER,  "white", "black")
            self.MarkerDefine(wxSTC_MARKNUM_FOLDERSUB,     wxSTC_MARK_VLINE,    "white", "black")
            self.MarkerDefine(wxSTC_MARKNUM_FOLDER,        wxSTC_MARK_BOXPLUS,  "white", "black")
            self.MarkerDefine(wxSTC_MARKNUM_FOLDEROPEN,    wxSTC_MARK_BOXMINUS, "white", "black")
            
            EVT_STC_MARGINCLICK(self, self.id, self.OnMarginClick)
        else:
            self.SetProperty("fold", "0")
            self.SetMarginWidth(2, 0)
            
    def OnMarginClick(self, event=None):
        # taken from the wxPython demo
        # fold and unfold as needed
        if event.GetMargin() == 2:
            if event.GetShift() and event.GetControl():
                self.FoldAll()
            else:
                lineClicked = self.LineFromPosition(event.GetPosition())
                if self.GetFoldLevel(lineClicked) & wxSTC_FOLDLEVELHEADERFLAG:
                    if event.GetShift():
                        self.SetFoldExpanded(lineClicked, true)
                        self.Expand(lineClicked, true, true, 1)
                    elif event.GetControl():
                        if self.GetFoldExpanded(lineClicked):
                            self.SetFoldExpanded(lineClicked, false)
                            self.Expand(lineClicked, false, true, 0)
                        else:
                            self.SetFoldExpanded(lineClicked, true)
                            self.Expand(lineClicked, true, true, 100)
                    else:
                        self.ToggleFold(lineClicked)

    def FoldAll(self):
        # taken from the wxPython demo
        lineCount = self.GetLineCount()
        expanding = true

        # find out if we are folding or unfolding
        for lineNum in range(lineCount):
            if self.GetFoldLevel(lineNum) & wxSTC_FOLDLEVELHEADERFLAG:
                expanding = not self.GetFoldExpanded(lineNum)
                break

        lineNum = 0
        while lineNum < lineCount:
            level = self.GetFoldLevel(lineNum)
            if level & wxSTC_FOLDLEVELHEADERFLAG and \
               (level & wxSTC_FOLDLEVELNUMBERMASK) == wxSTC_FOLDLEVELBASE:

                if expanding:
                    self.SetFoldExpanded(lineNum, true)
                    lineNum = self.Expand(lineNum, true)
                    lineNum = lineNum - 1
                else:
                    lastChild = self.GetLastChild(lineNum, -1)
                    self.SetFoldExpanded(lineNum, false)
                    if lastChild > lineNum:
                        self.HideLines(lineNum+1, lastChild)

            lineNum = lineNum + 1

    def Expand(self, line, doExpand, force=false, visLevels=0, level=-1):
        # taken from the wxPython demo
        lastChild = self.GetLastChild(line, level)
        line = line + 1
        while line <= lastChild:
            if force:
                if visLevels > 0:
                    self.ShowLines(line, line)
                else:
                    self.HideLines(line, line)
            else:
                if doExpand:
                    self.ShowLines(line, line)

            if level == -1:
                level = self.GetFoldLevel(line)

            if level & wxSTC_FOLDLEVELHEADERFLAG:
                if force:
                    if visLevels > 1:
                        self.SetFoldExpanded(line, true)
                    else:
                        self.SetFoldExpanded(line, false)
                    line = self.Expand(line, doExpand, force, visLevels-1)

                else:
                    if doExpand and self.GetFoldExpanded(line):
                        line = self.Expand(line, true, force, visLevels-1)
                    else:
                        line = self.Expand(line, false, force, visLevels-1)
            else:
                line = line + 1;

        return line


    def get_filename(self):
        """ Return filename for fileselector. """
        return self.filename or "<no filename>"
        
    # some editing stuff
    
    def delete_current_line(self):
        self.SendMsg(wxSTC_CMD_LINEDELETE)
        