# archiver.py

import date
import string

class ArchiverError(Exception): pass

class Archiver:
    """ Makes an "archive", a list of (key, value) pairs with dates for the
        keys and a list of blogentry ids for the values. (Obviously, these
        entries fall in the period signified by the date.) Weblog archives are
        based on this list.
    """

    def __init__(self, database):
        self.db = database

    def make_archive(self, method, visible_only=0):
        """ - d:X makes pages per X days, e.g. d:7 for weekly, or d:1 for daily
            - e:X makes pages for X entries, e.g. e:1 for a page per entry, or
              e:10 for 10 entries per page, etc.
            - m:1 makes pages per month (number is irrelevant).

            Returns a list of (key, value) pairs with dates as the keys and a
            list of entry ids for the values.
        """
        parts = string.split(method, ":") + ["1"]
        letter, number = parts[0], int(parts[1])
        # get all entry ids and dates
        entries = self.db.get_entry_dates(visible_only)

        if letter == "d":
            if number == 1:
                return self.make_archive_daily(entries)
            else:
                return self.make_archive_n_days(entries, number)
        elif letter == "e":
            return self.make_archive_n_entries(entries, number)
        elif letter == "m":
            return self.make_archive_monthly(entries)
        else:
            raise ArchiverError, "Unknown method"

    def make_archive_daily(self, entries):
        d = {}
        for entry in entries:
            archive_date = entry.insert_date[:10]
            lst = d.get(archive_date, [])
            lst.append(entry.blogentry_id)
            d[archive_date] = lst
        items = d.items()
        items.sort()
        return items

    def make_archive_n_days(self, entries, n):
        d = {}
        for entry in entries:
            # get the julian number of the insert_date
            da = date.Date(entry.insert_date)
            j = da.julian()
            # how many days from the beginning of an n-day period?
            diff = j % n
            # move that many days back for the archive date
            da.dec(diff)
            archive_date = da.isodate()[:10]
            lst = d.get(archive_date, [])
            lst.append(entry.blogentry_id)
            d[archive_date] = lst
        items = d.items()
        items.sort()
        return items

    def make_archive_monthly(self, entries):
        d = {}
        for entry in entries:
            archive_date = entry.insert_date[:7]    # keep YYYY-MM
            lst = d.get(archive_date, [])
            lst.append(entry.blogentry_id)
            d[archive_date] = lst
        items = d.items()
        items.sort()
        return items

    def make_archive_n_entries(self, entries, n):
        d = {}
        count = 1
        sublist = []
        for i in range(len(entries)-1,-1,-1):   # walk backwards
            if len(sublist) >= n:
                d[count] = sublist
                count = count + 1
                sublist = []
            sublist.append(entries[i].blogentry_id)
        if sublist:
            d[count] = sublist
        
        items = d.items()
        items.sort()
        return items

    ###

    def get_page_name(self, method, arch_id):
        if isinstance(arch_id, str):
            spam = string.replace(arch_id, "-", "_")
        elif isinstance(arch_id, int):
            spam = "%05d" % (arch_id,)
        eggs = string.replace(method, ":", "")

        return "arch_%s_%s.html" % (eggs, spam)


