# input_dialog.py

from Tkinter import *
#
import dialog
import resource

def get_value(widget):
    """ Get a value, as a string, from a widget. """
    if isinstance(widget, Text):
        return widget.get(1.0, END)
    elif isinstance(widget, Checkbutton):
        return str(widget.var.get())
 
def set_value(widget, value):
    """ Set a widget's value. 'value' must be a string. """
    if isinstance(widget, Text):
        widget.insert(END, value)
    elif isinstance(widget, Checkbutton):
        widget.var.set(int(value))

def set_tablink(widget1, widget2):
    """ Create a link between widget1 and widget2 so that pressing Tab in
        widget1 moves focus to widget2. """
    def link(event=None, target=widget2):
        target.focus_set()
        return "break"
    widget1.bind("<Tab>", link)

class InputDialog(dialog.Dialog):

    """ A standard dialog for editing (some of) an object's attributes. """

    prefix = "something_"   # override this in subclasses

    def __init__(self, master, dataobj, title):
        # dataobj must be a Configuration subclass, so it can be loaded and
        # saved.
        self.dataobj = dataobj
        self.data = {}

        dialog.Dialog.__init__(self, master, title)

    ###
    ### create widgets

    def create_inputbox(self, master, text, name, width=20, height=1):
        frame = Frame(master)
        label = Label(frame, text=text, width=22, justify=RIGHT, anchor=E,
         font=resource.SMALL_FONT)
        label.pack(side=LEFT, fill=X)
        inputbox = Text(frame, height=height, width=width, 
         font=resource.SMALL_FONT)
        inputbox.pack(side=LEFT, fill=X)
        setattr(self, self.prefix + name, inputbox)
        
        # set some attributes for later reference
        frame.inputbox = inputbox
        frame.label = label
       
        return frame

    def create_checkbox(self, master, text, name, comment=""):
        frame = Frame(master)
        label = Label(frame, text=text, width=22, justify=RIGHT, anchor=E,
         font=resource.SMALL_FONT)
        label.pack(side=LEFT, fill=X)
        var = IntVar()
        checkbox = Checkbutton(frame, text=comment, variable=var)
        checkbox.pack(side=LEFT, fill=X)
        checkbox.var = var
        setattr(self, self.prefix + name, checkbox)

        frame.checkbox = checkbox
        frame.label = label

        return frame

    def create_panel(self, master, height=5):
        pass

    ###

    def validate(self):
        # store results in dict; when dialog is destroyed, we cannot retrieve
        # this info from the widgets anymore
        # note that this only works for Text widgets...
        for fieldname in self.get_fieldnames():
            widget = getattr(self, self.prefix + fieldname)
            value = get_value(widget)
            value = value.rstrip()  # no trailing newlines etc
            self.data[fieldname] = value

        return 1    

    def load(self):
        """ Load dataobj data from database and put them in the appropriate
            widgets. """
        for fieldname in self.get_fieldnames():
            value = getattr(self.dataobj, fieldname).rstrip()
            widget = getattr(self, self.prefix + fieldname)
            set_value(widget, value)

    def save(self):
        """ Store data in input fields in a configuration file. """
        for fieldname in self.get_fieldnames():
            value = self.data[fieldname].rstrip()
            setattr(self.dataobj, fieldname, value)
        self.dataobj.save()

    def get_fieldnames(self):
        # everything that starts with "blog_", but strip that leading "blog_"
        # when returing the field names
        lp = len(self.prefix)
        fieldnames = [key[lp:] for key in self.__dict__.keys()
                      if key.startswith(self.prefix)]
        return fieldnames
    
